<?php

namespace App\Http\Controllers;

use App\Models\Infografia;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class InfografiaController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:ver-infografia|crear-infografia|editar-infografia|borrar-infografia', ['only' => ['index']]);
        $this->middleware('permission:crear-infografia', ['only' => ['create', 'store']]);
        $this->middleware('permission:editar-infografia', ['only' => ['edit', 'update']]);
        $this->middleware('permission:borrar-infografia', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $infografias = Infografia::orderBy('created_at', 'desc')->orderBy('anio', 'desc')->get(); // Cambia el número según tus necesidades
        return view('panel-infografias.index', compact('infografias'))->with('messsage', 'La infografía se ha subido correctamente');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('panel-infografias.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'infografia' => 'required|mimes:jpeg,png,svg,jpg|max:6144',
            'miniatura' => 'required|mimes:jpeg,png,svg,jpg|max:1024',
            'fecha_publicacion' => 'required|date',
        ], [
            'nombre.required' => 'El nombre de la infografía es requerido',
            'nombre.string' => 'El nombre debe ser una cadena de texto',
            'nombre.max' => 'El nombre no puede exceder los 255 caracteres',

            'infografia.required' => 'Se requiere la infografía',
            'infografia.mimes' => 'El archivo subido no es una imagen válida',
            'infografia.max' => 'La infografía supera el tamaño máximo de 6MB',

            'miniatura.mimes' => 'La miniatura no es una imagen válida',
            'miniatura.required' => 'No ha subido una miniatura',
            'miniatura.max' => 'La miniatura pesa más de 1MB',

            'fecha_publicacion.required' => 'La fecha de publicación es obligatoria.',
            'fecha_publicacion.date' => 'La fecha de publicación debe ser una fecha válida.',
        ]);


        $Infografia = $request->all();
        $Infografia['anio'] = \Carbon\Carbon::parse($request->input('fecha_publicacion'))->format('Y');

        if ($infografia = $request->file('infografia')) {
            $rutaGuardarInfografia = 'imagenes/infografias-carpeta/';
            $nombreInfografia = Str::slug($request->input('nombre')) . '-' . date('YmdHis') . '.' . $infografia->getClientOriginalExtension();
            $infografia->move($rutaGuardarInfografia, $nombreInfografia);
            $Infografia['infografia'] = $nombreInfografia;
        }

        if ($miniatura = $request->file('miniatura')) {
            $rutaGuardarMiniatura = 'imagenes/infografias-carpeta/miniaturas/';
            $nombreMiniatura = 'miniatura-' . Str::slug($request->input('nombre')) . '-' . date('YmdHis') . '.' . $miniatura->getClientOriginalExtension();
            $miniatura->move($rutaGuardarMiniatura, $nombreMiniatura);
            $Infografia['miniatura'] = $nombreMiniatura;
        }


        // Crear una nueva instancia del modelo Queja con los datos del formulario
        Infografia::create($Infografia);

        // Redirigir a la página de inicio o a donde lo desees después de guardar la queja
        // return redirect()->route('inicio.index')->with('success', '¡La queja se ha enviado exitosamente!');
        return redirect()->route('panel-infografias.index')->with('success', 'La infografía se ha actualizado correctamente');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Infografia  $infografia
     * @return \Illuminate\Http\Response
     */
    public function show(Infografia $infografia)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Infografia  $infografia
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $infografias = Infografia::findOrFail($id);
        return view('panel-infografias.editar', compact('infografias'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Infografia  $infografia
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'nombre' => 'required',
            'infografia' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:6144',
            'miniatura' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:1024',
            'fecha_publicacion' => 'required|date',
        ], [
            'nombre.required' => 'El nombre de la infografía es requerido',

            'miniatura.mimes' => 'El archivo no es una imagen válida',
            'miniatura.image' => 'El archivo no es una imagen válida',
            'miniatura.max' => 'La miniatura es más pesada de 1MB',

            'infografia.image' => 'El archivo no es una imagen válida',
            'infografia.max' => 'La infografía pesa más de 6MB',
            'infografia.mimes' => 'El archivo no es una imagen válida',

            'fecha_publicacion.required' => 'La fecha de publicación es obligatoria.',
            'fecha_publicacion.date' => 'La fecha de publicación debe ser una fecha válida.',
        ]);

        $infografia = Infografia::findOrFail($id);

        // Actualizar campos no relacionados con archivos
        $infografia->nombre = $request->nombre;
        $infografia->fecha_publicacion = $request->fecha_publicacion;
        $infografia['anio'] = \Carbon\Carbon::parse($request->input('fecha_publicacion'))->format('Y');  // Extrae y asigna el año

        // Manejar la actualización de la infografía (si se proporciona un nuevo archivo)
        if ($request->hasFile('infografia')) {
            // Eliminar archivo anterior
            if (file_exists(public_path('imagenes/infografias-carpeta/' . $infografia->infografia))) {
                unlink(public_path('imagenes/infografias-carpeta/' . $infografia->infografia));
            }

            // Guardar el nuevo archivo
            $file = $request->file('infografia');
            $nombre_slug = Str::slug($request->nombre);
            $nombre_documento = $nombre_slug . '_' . date('YmdHis') . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('imagenes/infografias-carpeta/'), $nombre_documento);
            $infografia->infografia = $nombre_documento;
        }

        // Manejar la actualización de la miniatura (si se proporciona un nuevo archivo)
        if ($request->hasFile('miniatura')) {
            // Eliminar archivo anterior
            if (file_exists(public_path('imagenes/infografias-carpeta/miniaturas/' . $infografia->miniatura))) {
                unlink(public_path('imagenes/infografias-carpeta/miniaturas/' . $infografia->miniatura));
            }

            // Guardar la nueva miniatura
            $file = $request->file('miniatura');
            $nombre_slug = Str::slug($request->nombre);
            $nombre_miniatura = 'miniatura-' . $nombre_slug . '_' . date('YmdHis') . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('imagenes/infografias-carpeta/miniaturas/'), $nombre_miniatura);
            $infografia->miniatura = $nombre_miniatura;
        }

        // Guardar los cambios en la base de datos
        $infografia->save();

        return redirect()->route('panel-infografias.index')->with('success', 'La infografía se ha actualizado correctamente');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Infografia  $infografia
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $infografia = Infografia::findOrFail($id);

        // Eliminar el archivo de infografía si existe
        $rutaInfografia  = public_path('imagenes/infografias-carpeta/') . $infografia->infografia;
        // Verificar si el archivo existe y luego eliminarlo
        if (file_exists($rutaInfografia)) {
            unlink($rutaInfografia);
        }

        // Eliminar el archivo de miniatura si existe
        $rutaMiniatura = public_path('imagenes/infografias-carpeta/miniaturas/') . $infografia->miniatura;
        if (file_exists($rutaMiniatura)) {
            unlink($rutaMiniatura);
        }

        // Eliminar el registro de la base de datos
        $infografia->delete();

        return redirect()->route('panel-infografias.index')->with('success', 'La infografía se ha borrado correctamente');
    }

    public function mostrarInfografias()
    {
        $fechaActual = \Carbon\Carbon::now();

        $infografias = Infografia::where(function ($query) use ($fechaActual) {
            $query->where('fecha_publicacion', '<=', $fechaActual)
                ->orWhereNull('fecha_publicacion'); // Incluye registros sin fecha_publicacion
        })
            ->orderBy('anio', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(12);

        return view('infografias', compact('infografias'));
    }
}
