<?php

namespace App\Http\Controllers;

use App\Models\Fotografia;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Pagination\LengthAwarePaginator;

class FotografiaController extends Controller
{
    public function __construct()
    {
        // Configura la localización global de Carbon a español
        Carbon::setLocale('es');
        $this->middleware('permission:ver-fotografia|crear-fotografia|editar-fotografia|borrar-fotografia', ['only' => ['index']]);
        $this->middleware('permission:crear-fotografia', ['only' => ['create', 'store']]);
        $this->middleware('permission:editar-fotografia', ['only' => ['edit', 'update']]);
        $this->middleware('permission:borrar-fotografia', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $fotografias = Fotografia::orderBy('created_at', 'desc')->orderBy('fecha', 'desc')->get();
        return view('panel-fotografias.index', compact('fotografias'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('panel-fotografias.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre_evento' => 'required|string|max:255',
            'nombre_archivo.*' => 'required|mimes:jpeg,png,svg,jpg,mp4,avi,mkv|max:51200',
            'fecha' => 'required|date',
        ], [
            'nombre_evento.required' => 'El nombre del evento es requerido',
            'nombre_archivo.*.required' => 'Se requiere al menos un medio',
            'nombre_archivo.*.mimes' => 'El archivo subido no es un formato válido (solo imágenes o videos)',
            'nombre_archivo.*.max' => 'El archivo supera el tamaño máximo permitido',
            'fecha.required' => 'Se requiere la fecha del evento',
            'fecha.date' => 'El formato de fecha no es válido',
        ]);

        $fotografias = [];
        $rutaGuardar = public_path('imagenes/fotos-carpeta/');

        // Obtener el número actual de archivos para el evento
        $numeroArchivosExistentes = Fotografia::where('nombre_evento', $request->nombre_evento)->count();

        if ($request->hasFile('nombre_archivo')) {
            foreach ($request->file('nombre_archivo') as $index => $archivo) {
                $nombreOriginal = pathinfo($archivo->getClientOriginalName(), PATHINFO_FILENAME);
                // Incrementar el contador para cada archivo
                $sufijo = $numeroArchivosExistentes + $index + 1;
                $nombreArchivo = Str::slug($request->nombre_evento) . '_' . $sufijo . '.' . $archivo->getClientOriginalExtension();

                // Mover el archivo a la carpeta de destino
                $archivo->move($rutaGuardar, $nombreArchivo);

                $fotografias[] = [
                    'nombre_evento' => $request->nombre_evento,
                    'nombre_archivo' => $nombreArchivo,
                    'fecha' => $request->fecha,
                ];
            }
        }

        // Guardar las fotografías en la base de datos
        Fotografia::insert($fotografias);

        return response()->json(['success' => 'Las fotografías se han subido correctamente.']);
    }


    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Fotografia  $fotografia
     * @return \Illuminate\Http\Response
     */
    public function show(Fotografia $fotografia)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Fotografia  $fotografia
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $fotos = Fotografia::findOrFail($id);
        return view('panel-fotografias.editar', compact('fotos'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Fotografia  $fotografia
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'nombre_evento' => 'required|string|max:255',
            'fecha' => 'required|date',
            'nombre_archivo' => 'nullable|mimes:jpeg,png,svg,jpg,mp4,avi,mkv|max:51200',
        ], [
            'nombre_evento.required' => 'El nombre del evento es requerido',
            'fecha.required' => 'La fecha del evento es requerida',
            'fecha.date' => 'El formato de la fecha no es válido',
            'nombre_archivo.mimes' => 'El archivo no es una imagen o video válida',
            'nombre_archivo.max' => 'El archivo debe ser menor a 50MB',
        ]);

        $fotos = Fotografia::findOrFail($id);

        // Actualizar campos no relacionados con archivos
        $fotos->nombre_evento = $request->nombre_evento;
        $fotos->fecha = $request->fecha;

        // Manejar la actualización de la ruta del documento (si se proporciona un nuevo archivo)
        if ($request->hasFile('nombre_archivo')) {
            $foto = $request->file('nombre_archivo');
            $nombre_foto = pathinfo($foto->getClientOriginalName(), PATHINFO_FILENAME);
            $nombre_documento = Str::slug($request->nombre_evento) . '_' . date('YmdHis') . '.' . $foto->getClientOriginalExtension();
            $foto->move(public_path('imagenes/fotos-carpeta/'), $nombre_documento);
            $fotos->nombre_archivo = $nombre_documento;
        }

        // Guardar los cambios en la base de datos
        $fotos->save();

        return redirect()->route('panel-fotografias.index')->with('success', 'La fotografía se ha actualizado correctamente.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Fotografia  $fotografia
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $foto = Fotografia::findOrFail($id);

        // Obtener la ruta del archivo de la foto
        $rutaArchivo = public_path('imagenes/fotos-carpeta/') . $foto->nombre_archivo;

        // Verificar si el archivo existe y luego eliminarlo
        if (file_exists($rutaArchivo)) {
            unlink($rutaArchivo);
        }

        // Eliminar el registro de la base de datos
        $foto->delete();

        return redirect()->route('panel-fotografias.index')->with('success', 'La fotografía se ha eliminado correctamente.');
    }

    public function mostrarFotos()
    {
        // Agrupa las fotos por mes y año, en español
        $fotos = DB::table('fotografias')
            ->select('nombre_evento', 'fecha', 'nombre_archivo')
            ->orderBy('nombre_evento')
            ->orderBy('nombre_archivo')
            ->get()
            ->groupBy(function ($foto) {
                return Carbon::parse($foto->fecha)->locale('es')->isoFormat('MMMM Y'); // Formato en español
            });
    
        // Obtener los meses únicos, en español
        $months = Fotografia::select(DB::raw('MONTH(fecha) as month'), DB::raw('YEAR(fecha) as year'))
            ->distinct()
            ->orderByRaw('YEAR(fecha) DESC, MONTH(fecha) DESC') // Usamos orderByRaw para evitar conflictos
            ->get()
            ->mapWithKeys(function ($item) {
                // Cambia el locale para obtener el mes en español y capitaliza la primera letra
                $month = ucfirst(Carbon::create()->month($item->month)->locale('es')->isoFormat('MMMM'));
                return [$item->month . '-' . $item->year => "$month de $item->year"];
            })
            ->toArray();
    
        // Obtener todas las fechas de las fotos
        $fechasFotos = Fotografia::orderBy('fecha', 'desc')->pluck('fecha');
    
        // Agrupar las fechas por mes y año
        $fechasAgrupadas = $fechasFotos->groupBy(function ($date) {
            return Carbon::parse($date)->format('F Y');
        });
    
        // Paginación manual
        $currentPage = request()->query('page', 1);
        $itemsPerPage = 1;
        $currentItems = $fechasAgrupadas->slice(($currentPage - 1) * $itemsPerPage, $itemsPerPage)->all();
    
        // Crear paginación manual
        $fechasPaginadas = new LengthAwarePaginator(
            $currentItems,
            count($fechasAgrupadas),
            $itemsPerPage,
            $currentPage,
            ['path' => url()->current()]
        );
        // Limitar la cantidad de enlaces visibles a cada lado del actual
        $fechasPaginadas->onEachSide(1);
    
        return view('galeria', compact('fotos', 'months', 'fechasPaginadas'));
    }
    

    public function mostrarFotosPorMes($mes)
    {
        // Divide el valor del mes y año
        [$month, $year] = explode('-', $mes);

        // Obtén el nombre del mes en español
        $monthName = \Carbon\Carbon::createFromFormat('m', $month)->locale('es')->monthName;

        // Filtra las fotografías por mes y año
        $fotos = DB::table('fotografias')
            ->select('nombre_evento', 'fecha', 'nombre_archivo')
            ->whereMonth('fecha', $month)
            ->whereYear('fecha', $year)
            ->orderBy('nombre_evento')
            ->orderBy('nombre_archivo')
            ->get();

        $groupedData_fotos = collect($fotos)->groupBy('nombre_evento');

        return view('fotos_por_mes', compact('groupedData_fotos', 'monthName', 'year'))->render();
    }
}
