<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use App\Models\Evento;
use Illuminate\Http\Request;
use Carbon\Carbon;

class EventoController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:ver-evento|crear-evento|editar-evento|borrar-evento', ['only' => ['index']]);
        $this->middleware('permission:crear-evento', ['only' => ['create', 'store']]);
        $this->middleware('permission:editar-evento', ['only' => ['edit', 'update']]);
        $this->middleware('permission:borrar-evento', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $eventos = Evento::all();
        $event = null;
        if ($request->has('event_id')) {
            $event = Evento::find($request->input('event_id'));
        }
        return view('panel-eventos.index', compact('eventos', 'event'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'required|string',
            'fecha_evento' => 'required|date',
            'subtitulo' => 'nullable|string|max:255'
        ], [
            'titulo.required' => 'El título es obligatorio.',
            'titulo.string' => 'El título debe ser una cadena de texto.',
            'titulo.max' => 'El título no puede tener más de 255 caracteres.',

            'descripcion.required' => 'La descripción es obligatoria.',
            'descripcion.string' => 'La descripción debe ser una cadena de texto.',

            'fecha_evento.required' => 'La fecha del evento es obligatoria.',
            'fecha_evento.date' => 'La fecha del evento debe ser una fecha válida.',

            'subtitulo.string' => 'El horario debe ser valido.',
            'subtitulo.max' => 'El horario no puede tener más de 255 caracteres.'
        ]);

        Evento::create($validatedData);
        return redirect()->route('panel-eventos.index')->with('success', 'Evento creado exitosamente.');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Evento  $evento
     * @return \Illuminate\Http\Response
     */
    public function show(Evento $evento)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Evento  $evento
     * @return \Illuminate\Http\Response
     */
    public function edit(Evento $evento)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Evento  $evento
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'required|string',
            'fecha_evento' => 'required|date',
            'subtitulo' => 'nullable|string|max:255'
        ], [
            'titulo.required' => 'El título es obligatorio.',
            'titulo.string' => 'El título debe ser una cadena de texto.',
            'titulo.max' => 'El título no puede tener más de 255 caracteres.',
            'descripcion.required' => 'La descripción es obligatoria.',
            'descripcion.string' => 'La descripción debe ser una cadena de texto.',
            'fecha_evento.required' => 'La fecha del evento es obligatoria.',
            'fecha_evento.date' => 'La fecha del evento debe ser una fecha válida.',
            'subtitulo.string' => 'El subtítulo debe ser una cadena de texto.',
            'subtitulo.max' => 'El subtítulo no puede tener más de 255 caracteres.'
        ]);

        $event = Evento::find($id);
        $event->update($validatedData);
        return redirect()->route('panel-eventos.index')->with('success', 'Evento actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Evento  $evento
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $event = Evento::find($id);
        $event->delete();
        return redirect()->route('panel-eventos.index')->with('success', 'Evento eliminado exitosamente.');
    }

    public function mostrarEventos()
    {
        Carbon::setLocale('es');  // Configurar Carbon para español

        // Agrupar eventos por mes y año en formato de fecha estándar
        $eventos = Evento::orderBy('fecha_evento', 'asc')->get()->groupBy(function ($date) {
            return Carbon::parse($date->fecha_evento)->format('Y-m'); // Agrupar por año y mes
        });

        $mesActual = Carbon::now()->format('Y-m');

        if (!$eventos->has($mesActual)) {
            // Encuentra el mes más cercano anterior que tenga eventos
            $mesActual = $eventos->keys()->filter(function ($mes) {
                return Carbon::parse($mes)->lessThanOrEqualTo(Carbon::now());
            })->last();
        }

        $blogs = Blog::where('is_published', 1)
            ->whereDate('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(6);

        return view('eventos', compact('eventos', 'mesActual', 'blogs'));
    }
}
