<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;

class BlogController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:ver-blog|crear-blog|editar-blog|borrar-blog', ['only' => ['index']]);
        $this->middleware('permission:crear-blog', ['only' => ['create', 'store']]);
        $this->middleware('permission:editar-blog', ['only' => ['edit', 'update']]);
        $this->middleware('permission:borrar-blog', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $blogs = Blog::orderBy('published_at', 'desc')->orderBy('created_at', 'desc')->get();
        return view('panel-blogs.index', compact('blogs'))->with('messsage', 'El blog se ha subido correctamente');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('panel-blogs.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Validar los datos del formulario
        $request->validate([
            'titulo' => 'required|string|max:255',
            'contenido' => 'required|string',
            'extracto' => 'required|string',
            'imagen_1' => 'image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_2' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_3' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_4' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_5' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'is_published' => 'boolean',
            'published_at' => 'required|date'
        ], [
            // Mensajes de validación personalizados
            'titulo.required' => 'El título del blog es requerido',
            'contenido.required' => 'El contenido del blog es requerido',
            'extracto.required' => 'El extracto del blog es requerido',
            'imagen_1.image' => 'El archivo de la imagen 1 debe ser una imagen válida',
            'imagen_1.mimes' => 'La imagen 1 debe ser un archivo de tipo: jpeg, png, svg, jpg',
            'imagen_1.max' => 'La imagen 1 no puede ser mayor a 2048 KB',
            'is_published.boolean' => 'El campo "publicado" debe ser un valor booleano',
            'published_at.required' => 'Se requiere una fecha de publicación del blog',
            'published_at.date' => 'El campo debe ser fecha'
        ]);

        // Generar el slug a partir del título
        $slug = Str::slug($request->titulo);

        // Verificar si el slug ya existe en la base de datos y añadir un sufijo numérico si es necesario
        $suffix = 1;
        while (Blog::where('slug', $slug)->exists()) {
            $slug = Str::slug($request->titulo) . '-' . $suffix++;
        }

        // Crear una nueva instancia de Blog con los datos del formulario
        $blog = new Blog();
        $blog->titulo = $request->titulo;
        $blog->slug = $slug;
        $blog->contenido = $request->contenido;
        $blog->extracto = $request->extracto;
        $blog->published_at = $request->published_at;
        $blog->is_published = $request->input('is_published');

        // Manejo de imágenes
        $rutaGuardarImagen = 'imagenes/blogs-carpeta/';
        $timestamp = time();

        // Imagen 1 - Miniatura
        if ($imagen = $request->file('imagen_1')) {
            $imagenNombre = 'Miniatura_Blog_' . $slug . '_' . $timestamp . '.' . $imagen->getClientOriginalExtension();
            $imagen->move($rutaGuardarImagen, $imagenNombre);
            $blog->imagen_1 = $rutaGuardarImagen . $imagenNombre;
        }

        // Imágenes adicionales (2 a 5)
        foreach (range(2, 5) as $index) {
            if ($imagen = $request->file('imagen_' . $index)) {
                $imagenNombre = 'Imagen_Adicional_Blog_' . ($index - 1) . '_' . $slug . '_' . $timestamp . '.' . $imagen->getClientOriginalExtension();
                $imagen->move($rutaGuardarImagen, $imagenNombre);
                $blog->{'imagen_' . $index} = $rutaGuardarImagen . $imagenNombre;
            }
        }

        // Guardar el blog en la base de datos
        $blog->save();

        // Redireccionar a la vista index con un mensaje de éxito
        return redirect()->route('blogs.index')->with('success', 'El blog se ha subido correctamente');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function show($slug)
    {
        $blog = Blog::where('slug', $slug)->firstOrFail();
        $imagenes = [];
        for ($i = 2; $i <= 5; $i++) {
            $campo = 'imagen_' . $i;
            if (!empty($blog->$campo)) {
                $imagenes[] = $blog->$campo;
            }
        }

        $recentBlogs = Blog::where('published_at', '<=', Carbon::now())
            ->where('is_published', true)
            ->orderBy('published_at', 'desc')
            ->take(3)
            ->get();

        return view('blog', compact('blog', 'imagenes', 'recentBlogs'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $blogs = Blog::findOrFail($id);
        return view('panel-blogs.editar', compact('blogs'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // Validar los datos del formulario
        $request->validate([
            'titulo' => 'required|string|max:255',
            'contenido' => 'required|string',
            'extracto' => 'required|string',
            'imagen_1' => 'image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_2' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_3' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_4' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'imagen_5' => 'nullable|image|mimes:jpeg,png,svg,jpg|max:2048',
            'is_published' => 'boolean',
            'published_at' => 'required|date'
        ], [
            // Mensajes de validación personalizados
            'titulo.required' => 'El título del blog es requerido',
            'contenido.required' => 'El contenido del blog es requerido',
            'extracto.required' => 'El extracto del blog es requerido',
            'imagen_1.image' => 'El archivo de la imagen 1 debe ser una imagen válida',
            'imagen_1.mimes' => 'La imagen 1 debe ser un archivo de tipo: jpeg, png, svg, jpg',
            'imagen_1.max' => 'La imagen 1 no puede ser mayor a 2048 KB',
            'is_published.boolean' => 'El campo "publicado" debe ser un valor booleano',
            'published_at.required' => 'Se requiere una fecha de publicación del blog',
            'published_at.date' => 'El campo debe ser una fecha válida'
        ]);

        // Encontrar el blog a actualizar
        $blog = Blog::findOrFail($id);

        // Generar el slug a partir del título
        $slug = Str::slug($request->titulo);

        // Actualizar los atributos del blog con los datos del formulario
        $blog->titulo = $request->titulo;
        $blog->slug = $slug;
        $blog->contenido = $request->contenido;
        $blog->extracto = $request->extracto;
        $blog->published_at = $request->published_at;
        $blog->is_published = $request->input('is_published');

        // Ruta para guardar las imágenes
        $rutaGuardarImagen = 'imagenes/blogs-carpeta/';
        $timestamp = time();

        // Subir y guardar la imagen 1 (Miniatura)
        if ($imagen = $request->file('imagen_1')) {
            $imagenNombre = 'Miniatura_Blog_' . $slug . '_' . $timestamp . '.' . $imagen->getClientOriginalExtension();
            $imagen->move($rutaGuardarImagen, $imagenNombre);
            $blog->imagen_1 = $rutaGuardarImagen . $imagenNombre;
        }

        // Subir y guardar imágenes 2 a 5 (Imágenes adicionales)
        foreach (range(2, 5) as $index) {
            if ($request->input('eliminar_imagen_' . $index) == '1') {
                // Si se selecciona eliminar la imagen, se elimina de la base de datos
                $blog->{'imagen_' . $index} = null;
            } elseif ($imagen = $request->file('imagen_' . $index)) {
                // Renombrar y mover la imagen
                $imagenNombre = 'Imagen_Adicional_Blog_' . ($index - 1) . '_' . $slug . '_' . $timestamp . '.' . $imagen->getClientOriginalExtension();
                $imagen->move($rutaGuardarImagen, $imagenNombre);
                $blog->{'imagen_' . $index} = $rutaGuardarImagen . $imagenNombre;
            }
        }

        // Guardar los cambios en la base de datos
        $blog->save();

        // Redireccionar a la vista index con un mensaje de éxito
        return redirect()->route('panel-blogs.index')->with('success', 'El blog se ha actualizado correctamente');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // Encontrar el blog por su ID
        $blog = Blog::findOrFail($id);

        // Array con los nombres de las columnas de las imágenes
        $imagenes = ['imagen_1', 'imagen_2', 'imagen_3', 'imagen_4', 'imagen_5'];

        // Eliminar las imágenes asociadas si existen
        foreach ($imagenes as $imagen) {
            if ($blog->$imagen) {
                $rutaImagen = public_path($blog->$imagen);
                if (file_exists($rutaImagen)) {
                    unlink($rutaImagen); // Eliminar la imagen del sistema de archivos
                }
            }
        }

        // Eliminar el registro de la base de datos
        $blog->delete();

        // Redireccionar a la vista index con un mensaje de éxito
        return redirect()->route('panel-blogs.index')->with('message', 'El blog se ha eliminado correctamente');
    }

    public function mostrarBlogs($slug)
    {
        $blog = Blog::where('slug', $slug)->firstOrFail();
        $imagenes = [];
        for ($i = 2; $i <= 5; $i++) {
            $campo = 'imagen_' . $i;
            if (!empty($blog->$campo)) {
                $imagenes[] = $blog->$campo;
            }
        }

        $recentBlogs = Blog::where('published_at', '<=', Carbon::now())
            ->where('is_published', true)
            ->orderBy('published_at', 'desc')
            ->take(3)
            ->get();

        return view('blog', compact('blog', 'imagenes', 'recentBlogs'));
    }
}
